#include "stdafx.h"
#include "DffExp.h"
#include "utilities.h"
#include <rpmorph.h>
#include "RwExpError.h"

#include "bezier.h"

typedef struct _GeomSort GeomSort;

struct _GeomSort
{
    RpGeometry         *geom;
    RwInt32            *vertexMap;
    RwInt32             currentIndex;
};

static RpMaterial  *
BuildVertexMap(RpMaterial * material, void *pData)
{
    GeomSort           *gs = (GeomSort *) pData;
    RwInt32             numTriangles = RpGeometryGetNumTriangles(gs->geom);
    RpTriangle         *tris = RpGeometryGetTriangles(gs->geom);
    RwInt32             i, j;

    for (i = 0; i < numTriangles; i++)
    {
        if (RpGeometryGetMaterial(gs->geom, tris[i].matIndex) == material)
        {
            for (j = 0; j < 3; j++)
            {
                if (gs->vertexMap[tris[i].vertIndex[j]] == -1)
                {
                    gs->vertexMap[tris[i].vertIndex[j]] = gs->currentIndex;
                    gs->currentIndex++;
                }
            }
        }
    }

    return (material);
}

RwInt32            *
GenerateVertexMapByMaterial(RpGeometry * geometry)
{
    GeomSort            gs;
    int                 i, j, tri;
    RwInt32             numVertices = RpGeometryGetNumVertices(geometry);
    RwInt32            *vertexMap = (RwInt32 *)

        RwMalloc(sizeof(RwInt32) * numVertices);
    RwInt32             numTris = RpGeometryGetNumTriangles(geometry);
    RpTriangle         *tris = RpGeometryGetTriangles(geometry);

    for (i = 0; i < numVertices; i++)
    {
        vertexMap[i] = -1;
    }
    gs.geom = geometry;
    gs.vertexMap = vertexMap;
    gs.currentIndex = 0;

    RpGeometryForAllMaterials(geometry, BuildVertexMap, &gs);

    for (tri = 0; tri < numTris; tri++)
    {
        for (j = 0; j < 3; j++)
        {
            tris[tri].vertIndex[j] =
                (RwUInt16) vertexMap[tris[tri].vertIndex[j]];
        }
    }

    return (vertexMap);
}

RpGeometry         *
DFFExport::CreateGeometryFromMesh(Mesh * mesh, Mtl * nodeMaterial)
{
    int
        nFlags = 0;
    int
        nNumNonDegenerateFaces = 0;
    int
        nNumVertices;

    BOOL textured;
    RpGeometry         *
        geometry = NULL;

    /* setup all the flags */
    if (m_exportNormals)
    {
        nFlags |= rpGEOMETRYNORMALS;
    }

    if (m_exportLit)
    {
        nFlags |= rpGEOMETRYLIGHT;
    }

    if (m_preLightClump)
    {
        nFlags |= rpGEOMETRYPRELIT;
    }

    if (m_colorVertexPrelight && mesh->vcFace)
    {
        nFlags |= rpGEOMETRYPRELIT;
    }

    nNumNonDegenerateFaces =
        NumNonDegenerateFaces(mesh, nodeMaterial, m_weldThreshold,
                              m_2SidedMaterials, m_weldVertices);
    if (nNumNonDegenerateFaces == 0)
    {
        return (NULL);
    }

    nNumVertices = m_remapper.GetNumVertices();

    /* is object textured */
    textured = (mesh->tvFace && (mesh->getNumTVerts() > 0));
    if (textured)
    {
        nFlags |= rpGEOMETRYTEXTURED;
    }

    /* should it be flagged as tri-stripable */
    if (m_triStripMesh)
    {
        nFlags |= rpGEOMETRYTRISTRIP;
    }

    //Renderware Geometries can save only a maximum of 65535 vertices
    //RpGeometryCreate will assert (and later crash) if we don't avoid calling it!
    if (nNumVertices > 65535)
    {
        return 0;
    }

    if (!
        (geometry =
         RpGeometryCreate(nNumVertices, nNumNonDegenerateFaces, nFlags)))
    {
        return (FALSE);
    }

    return (geometry);
}

void
DFFExport::SetGeometryTrianglesFromMesh(INode * node, Mesh * mesh,
                                        RpGeometry * geometry)
{
    RpTriangle         *
        triangles = RpGeometryGetTriangles(geometry);
    RpMaterial         *
        material = NULL;
    int
        faceNum = 0;
    int
        nNumFaces = mesh->getNumFaces();
    int
        lastMatID = -1;

    BOOL mirrored = node->GetObjectTM(0).Parity();

    for (faceNum = 0; faceNum < m_remapper.GetNumFaces(); faceNum++)
    {
        ReMapperFaceLoop   *
            remapFace;

        remapFace = m_remapper.GetFace(faceNum);

        if (!mirrored)
        {
            RpGeometryTriangleSetVertexIndices(geometry, triangles,
                                               (RwUInt16) (remapFace->index
                                                           [0]),
                                               (RwUInt16) (remapFace->index
                                                           [1]),
                                               (RwUInt16) (remapFace->index
                                                           [2]));
        }
        else
        {
            RpGeometryTriangleSetVertexIndices(geometry, triangles,
                                               (RwUInt16) (remapFace->index
                                                           [1]),
                                               (RwUInt16) (remapFace->index
                                                           [0]),
                                               (RwUInt16) (remapFace->index
                                                           [2]));
        }

        /* If we've moved to a new material see if it's already in the
         * geometry, if so reuse it (but check it's mask). If not create
         * a new one */
        if (mesh->faces[remapFace->faceNum].getMatID() != lastMatID)
        {
            matSearch info;

            info.material = NULL;
            info.maxMaterial = node->GetMtl();
            info.maxMatID = mesh->faces[remapFace->faceNum].getMatID();
            info.nameCase = m_texturenameCase;

            RpGeometryForAllMaterials(geometry, CompareRpMaterial2MaxMaterial,
                                      (void *) &info);

            if (info.material == NULL)
            {
                material =
                    Max2RpMaterial(node->GetMtl(), info.maxMatID,
                                   m_texturenameCase);
                RpGeometryTriangleSetMaterial(geometry, triangles, material);
                RpMaterialDestroy(material);
                m_WarningsList.add(
                    CWarning(wtInformational,node->GetName(),
                    "Added new material"));
            }
            else
            {
                material = info.material;
                CheckMaterialMask(material, info.maxMaterial, info.maxMatID,
                                  m_texturenameCase);
                RpGeometryTriangleSetMaterial(geometry, triangles, material);
            }
            lastMatID = info.maxMatID;
        }
        else
        {
            RpGeometryTriangleSetMaterial(geometry, triangles, material);
        }
        triangles++;
    }
}

void
DFFExport::ExtractGeometryFromNode(INode * node, RpClump * clump,
                                   RwFrame * inFrame, RpAtomic * &outAtomic,
                                   RwInt32 * &outVertexMap,
                                   Matrix3 &transform )
{
    //Get vertices, morph targets, polygons, normals, lighting & textured coords out of MAX node,
    //return it as RpGeometry stuffed into a new RpAtomic, atomic is given input frame,
    //and is finally added to clump!
    //vertices & normals are transformed by given transform matrix.
    //Also returns a vertex map - remember to RwFree vertex map!
    // e.g.
    //        if (vertexMap)
    //        {
    //            RwFree(vertexMap);
    //        }

    outAtomic = NULL;
    outVertexMap = NULL;

    //we only build geometry for visible nodes that can be converted to tri meshes.
    Object             *
        object = node->EvalWorldState(m_tvStart).obj;

    if (!object || !object->IsRenderable() || !node->Renderable()
        || node->IsNodeHidden()
        || !object->CanConvertToType(triObjectClassID))
    {
        return;
    }

    //get a tri mesh representing this node from MAX
    TriObject          *
        triObject = (TriObject *) object->ConvertToType(0, triObjectClassID);
    Mesh               *
        mesh = &triObject->mesh;

    // remap for both texture vertices and smoothing groups
    // free remap structure
    m_remapper.ResetReMap();
    m_remapper.DoRemap(mesh, node->GetMtl(), m_limitUVs, m_limitUVMax,
                       m_weldThreshold, m_weldThresholdUV, m_2SidedMaterials,
                       m_backfaceNormals, m_weldVertices, &m_WarningsList);

    RpGeometry         *
        geometry = CreateGeometryFromMesh(mesh, node->GetMtl());

    if (geometry)
    {
        SetGeometryTrianglesFromMesh(node, mesh, geometry);
        // now the triangles and materials are set up
        //   build a vertex map sorted on material
        outVertexMap = GenerateVertexMapByMaterial(geometry);

        //do we want some morph targets?
        //if not just do one key frame
        //(which also currently seems to do lighting, normals, texture coords etc!)
        int
            numKeyFrames = 1;

        if (m_morphTargets)
        {
            //Count up some more keys to take for morph targets
            //This will go wrong when used in conjunction with skinned animations,
            //but one day we'd like it to work.
            //(so user shouldn't check welding + skinning at same time!!!)
            Interval v;

            v = object->ObjectValidity(m_tvStart);
            if (v.Start() > m_tvStart || v.End() < m_tvEnd)
            {
                numKeyFrames = m_nNumFrames / m_nAnimFrameInterval;
                if (numKeyFrames % m_nAnimFrameInterval)
                {
                    numKeyFrames++;
                }
            }
        }

        SetGeometryKeyFramesFromNode(node, geometry, numKeyFrames,
                                     outVertexMap, transform );

        RpGeometryUnlock(geometry);

        // check for coloured materials in the geometry
        if (GeometryHasColoredMaterials(geometry))
        {
            RpGeometrySetFlags(geometry,
                               RpGeometryGetFlags(geometry) |
                               rpGEOMETRYMODULATEMATERIALCOLOR);
        }

        outAtomic = RpAtomicCreate();
        if (outAtomic)
        {
            RpAtomicSetFrame(outAtomic, inFrame);
            RpAtomicSetGeometry(outAtomic, geometry, 0);
            RpGeometryDestroy(geometry);
            RpClumpAddAtomic(clump, outAtomic);

            if (GeometryHasMaterialEffects(geometry))
            {
                RpMatFXAtomicEnableEffects(outAtomic);
            }
            /* Add some export information to the warnings. */
            char temp[256];
            sprintf(temp,"Added atomic : Vertices=%i Triangles=%i Morph Targets=%i",
                geometry->numVertices,geometry->numTriangles,geometry->numMorphTargets);
            m_WarningsList.add(CWarning(wtInformational,node->GetName(),temp));
            /* Update global counts. */
            m_WarningsList.m_AtomicCount+=1;
            m_WarningsList.m_VertexCount+=geometry->numVertices;
            m_WarningsList.m_TriangleCount+=geometry->numTriangles;
        }
        else
        {
            RpGeometryDestroy(geometry);
        }
    }

    if (triObject != object)
    {
        triObject->DeleteMe();
    }

    //Renderware Geometries can save only a maximum of 65535 vertices
    //let's explain WHY export didn't work!
    if (m_remapper.GetNumVertices() > 65535)
    {
        char
            msg[10000];

        sprintf(msg,
                "%s has too many vertices (%d). Renderware can only handle a maximum of 65535 vertices per object",
                node->GetName(), m_remapper.GetNumVertices());

        throw RwExpError(msg);
    }
}

#define TICKSTOSECS(ticks) ((float)(ticks) / (float)TIME_TICKSPERSEC)

void
DFFExport::SetGeometryKeyFramesFromNode(INode * node,
                                        RpGeometry *
                                        geometry,
                                        int numKeyFrames, RwInt32 * vertexMap,
                                        //transform vertices & normals by this matrix
                                        //(we remove scale & translation before doing normals)
                                        Matrix3 & transform
                                        )
{
    int numVertices;
    int frameNum;

    TimeValue tvCurrent = m_tvStart;
    TimeValue tvPrev = m_tvStart;
    TimeValue
        tvKeyFrameInterval =
        (TimeValue) (m_nAnimFrameInterval * GetTicksPerFrame());
    RwTexCoords        *
        textureVerts = RpGeometryGetVertexTexCoords(geometry, rwTEXTURECOORDINATEINDEX0);
    RwRGBA             *
        preLight = NULL;

    RwBool textured = FALSE;

    if (numKeyFrames > 1)
    {
        RpMorphGeometryCreateInterpolators(geometry, numKeyFrames - 1);
    }

    for (frameNum = 0; frameNum < numKeyFrames; frameNum++)
    {
        RpMorphTarget      *
            keyFrame;

        if (frameNum > 0)
        {
            RpGeometryAddMorphTarget(geometry);
            RpMorphGeometrySetInterpolator(geometry, frameNum - 1,
                                           frameNum - 1, frameNum,
                                           TICKSTOSECS(tvCurrent - tvPrev));
        }

        keyFrame = RpGeometryGetMorphTarget(geometry, frameNum);
        if (keyFrame)
        {
            Object             *
                object;
            TriObject          *
                triObject;

            object = node->EvalWorldState(tvCurrent).obj;
            triObject =
                (TriObject *) object->ConvertToType(tvCurrent,
                                                    triObjectClassID);
            if (triObject)
            {
                int
                    vertexNum;

                RwSphere sphere;
                Mesh               *
                    mesh = &triObject->mesh;
                RwV3d              *
                    vertices = RpMorphTargetGetVertices(keyFrame);
                RwV3d              *
                    vnormals = NULL;

                if (m_exportNormals)
                {
                    vnormals = RpMorphTargetGetVertexNormals(keyFrame);
                    mesh->buildNormals();
                }

                textured |= (mesh->tvFace && (mesh->getNumTVerts() > 0));

                numVertices = m_remapper.GetNumVertices();
                if (frameNum == 0)
                {
                    if (m_preLightClump
                        || (m_colorVertexPrelight && mesh->vcFace))
                    {
                        preLight = RpGeometryGetPreLightColors(geometry);
                    }

                    for (vertexNum = 0; vertexNum < numVertices; vertexNum++)
                    {
                        ReMapperVertex     *
                            remapVertex;

                        /* Retrieve the appropriate vertex */
                        remapVertex = m_remapper.GetVertex(vertexNum);
                        if (textured)
                        {
                            textureVerts[vertexMap[vertexNum]].u =
                                remapVertex->u;
                            textureVerts[vertexMap[vertexNum]].v =
                                remapVertex->v;}

                        if (m_colorVertexPrelight && mesh->vcFace)
                        {
                            preLight[vertexMap[vertexNum]].red =
                                (RwUInt8) (mesh->vertCol
                                           [remapVertex->colorIndex].x *
                                           255.0f);
                            preLight[vertexMap[vertexNum]].green =
                                (RwUInt8) (mesh->vertCol
                                           [remapVertex->colorIndex].y *
                                           255.0f);
                            preLight[vertexMap[vertexNum]].blue =
                                (RwUInt8) (mesh->vertCol
                                           [remapVertex->colorIndex].z *
                                           255.0f);
                            preLight[vertexMap[vertexNum]].alpha = 255;
                        }
                        else if (m_preLightClump)
                        {
                            preLight[vertexMap[vertexNum]].red = 255;
                            preLight[vertexMap[vertexNum]].green = 255;
                            preLight[vertexMap[vertexNum]].blue = 255;
                            preLight[vertexMap[vertexNum]].alpha = 255;
                        }
                    }
                }

                //silly to transform normals with scaling & translation, so remove it
                Matrix3 normalsTransform = transform;
                normalsTransform.NoScale();
                normalsTransform.NoTrans();

                for (vertexNum = 0; vertexNum < numVertices; vertexNum++)
                {
                    ReMapperVertex     *
                        remapVertex;

                    Point3 maxVert;

                    /* Retrieve the appropriate vertex */
                    remapVertex = m_remapper.GetVertex(vertexNum);
                    maxVert = mesh->getVert(remapVertex->vertexIndex);
                    /* Apply the object offset then the scaling then user scale */
                    maxVert = maxVert * transform;
                    vertices[vertexMap[vertexNum]].x = maxVert.x;
                    vertices[vertexMap[vertexNum]].y = maxVert.y;
                    vertices[vertexMap[vertexNum]].z = maxVert.z;
                    /* find the normal */
                    if (m_exportNormals)
                    {
                        Point3 normal;

                        if (remapVertex->smGroup != 0)
                        {
                            RVertex            *
                                rvert =

                                mesh->getRVertPtr(remapVertex->vertexIndex);

                            if ((rvert->rFlags & NORCT_MASK) > 1)
                            {
                                unsigned int
                                    i;

                                for (i = 0; i < (rvert->rFlags & NORCT_MASK);
                                     i++)
                                {
                                    if (rvert->ern[i].getSmGroup() ==
                                        remapVertex->smGroup)
                                    {
                                        rvert->ern[i].normalize();
                                        normal = rvert->ern[i].getNormal();
                                        break;
                                    }
                                }
                            }
                            else
                            {
                                rvert->rn.normalize();
                                normal = rvert->rn.getNormal();
                            }
                        }
                        else
                        {
                            normal =
                                mesh->getFaceNormal(remapVertex->faceIndex);
                        }

                        normal = normal * normalsTransform;
                        if (remapVertex->backface)
                        {
                            normal *= -1.0f;
                        }
                        vnormals[vertexMap[vertexNum]].x = normal.x;
                        vnormals[vertexMap[vertexNum]].y = normal.y;
                        vnormals[vertexMap[vertexNum]].z = normal.z;
                    }
                }

                /* set bounding sphere */
                RpMorphTargetCalcBoundingSphere(keyFrame, &sphere);
                RpMorphTargetSetBoundingSphere(keyFrame, &sphere);
            }
        }
        tvPrev = tvCurrent;
        tvCurrent += tvKeyFrameInterval;
        if (tvCurrent > m_tvEnd)
        {
            tvCurrent = m_tvEnd;
        }
    }
}
